using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Audio;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.GamerServices;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Input;
using Microsoft.Xna.Framework.Media;
using Microsoft.Xna.Framework.Net;
using Microsoft.Xna.Framework.Storage;

namespace BreadAndCheesePong {

    public interface ISpriteBasedGame
    {
        void SetMessage(string message);
        GamePadState GetGamePad(PlayerIndex player);
        void StartGame();
        void EndGame();
    }

#region Sprajty

    public class BaseSprite {
        protected Texture2D spriteTexture;
        protected Rectangle spriteRectangle;

        /// <summary>
        /// aduje tekstur uywan w ramach danego sprajtu.
        /// </summary>
        /// <param name="inSpriteTexture">Uywana tekstura.</param>
        public void LoadTexture(Texture2D inSpriteTexture) {
            spriteTexture = inSpriteTexture;
        }

        /// <summary>
        /// aduje prostokt wyznaczajcy wymiary na potrzeby 
        /// operacji rysowania tego sprajtu ta.
        /// </summary>
        /// <param name="inSpriteRectangle">Prostokt, ktry ma zosta uyty.</param>
        public void SetRectangle(Rectangle inSpriteRectangle) {
            spriteRectangle = inSpriteRectangle;
        }

        /// <summary>
        /// Metoda wywoywana w momencie uruchamiania gry
        /// </summary>
        public virtual void StartGame() {
        }

        /// <summary>
        /// Metoda wywoywana w momencie zatrzymywania gry
        /// </summary>
        public virtual void EndGame() {
        }

        /// <summary>
        /// Rysuje tekstur na podstawie prostokta okrelajcego jej wymiary.
        /// </summary>
        /// <param name="spriteBatch">Obiekt klasy SpriteBatch, ktry ma zosta uyty
        /// przez operacj rysowania.</param>
        public virtual void Draw(SpriteBatch spriteBatch) {
            spriteBatch.Draw(spriteTexture, spriteRectangle, Color.White);
        }

        /// <summary>
        /// Aktualizuje zachowanie dla tytuu. Sprajt bazowy nie zawiera adnego
        /// zachowania aktualizacji, ale pozostae sprajty mog zawiera takie zachowania.
        /// </summary>
        /// <param name="game">Gra, ktrej przebieg ma by kontrolowany.</param>
        public virtual void Update(ISpriteBasedGame game) {
        }

        public BaseSprite(Texture2D inSpriteTexture, Rectangle inRectangle) {
            LoadTexture(inSpriteTexture);
            SetRectangle(inRectangle);
        }

    }

    public class TitleSprite : BaseSprite {
        /// <summary>
        /// Aktualizuje zachowanie dla tytuu. Metoda testuje stan przycisku A i uruchamia gr,
        /// jeli ten przycisk jest wcinity.
        /// </summary>
        /// <param name="game">Gra, ktrej przebieg ma by kontrolowany.</param>
        public override void Update(ISpriteBasedGame game) {
            if (game.GetGamePad(PlayerIndex.One).Buttons.A == ButtonState.Pressed) {
                game.StartGame();
            }
        }

        public TitleSprite(Texture2D inSpriteTexture, Rectangle inRectangle)
        : base(inSpriteTexture, inRectangle) {
        }
    }

    public class MovingSprite : BaseSprite {
        protected float x;
        protected float y;
        protected float xSpeed;
        protected float ySpeed;

        protected float initialX;
        protected float initialY;

        protected float minDisplayX;
        protected float maxDisplayX;

        protected float minDisplayY;
        protected float maxDisplayY;

        float displayWidth;
        float displayHeight;

        public float XPos
        {
            get
            {
                return x;
            }
        }

        public float YPos
        {
            get
            {
                return y;
            }
        }


        /// <summary>
        /// Sprawdza ewentualne kolizje pomidzy danym sprajtem a pozostaymi obiektami.
        /// </summary>
        /// <param name="target">Prostokt reprezentujcy pooenie innego obiektu.</param>
        /// <returns>true, jeli chleb koliduje z danym obiektem</returns>
        public virtual bool CheckCollision(Rectangle target) {
            return spriteRectangle.Intersects(target);
        }

        /// <summary>
        /// Tworzy poruszajcy si sprajt
        /// </summary>
        /// <param name="inSpriteTexture">Tekstura uywana dla tego sprajtu</param>
        /// <param name="widthFactor">Rozmiar obiektu w grze wyraony jako uamek szerokoci
        /// ekranu.</param>
        /// <param name="ticksToCrossScreen">Szybko poruszania si obiektu wyraona w liczbie taktw (1/60 sekundy) 
        /// potrzebnych do pokonania caego ekranu.</param>
        /// <param name="inMinDisplayX">minimalna warto XPos</param>
        /// <param name="inMaxDisplayX">maksymalna warto XPos</param>
        /// <param name="inMinDisplayY">minimalna warto YPos</param>
        /// <param name="inMaxDisplayY">maksymalna warto YPos</param>
        /// <param name="initialX">pocztkowa pozycja XPos paki</param>
        /// <param name="initialY">pocztkowa pozycja YPos paki</param>
        public MovingSprite(
                Texture2D inSpriteTexture,
                float widthFactor,
                float ticksToCrossScreen,
                float inMinDisplayX,
                float inMaxDisplayX,
                float inMinDisplayY,
                float inMaxDisplayY,
                float inInitialX,
                float inInitialY)
        : base(inSpriteTexture, Rectangle.Empty) {
            minDisplayX = inMinDisplayX;
            minDisplayY = inMinDisplayY;
            maxDisplayX = inMaxDisplayX;
            maxDisplayY = inMaxDisplayY;
            initialX = inInitialX;
            initialY = inInitialY;

            displayWidth = maxDisplayX - minDisplayX;
            displayHeight = maxDisplayY - minDisplayY;

            spriteRectangle.Width = (int) ((displayWidth * widthFactor) + 0.5f);
            float aspectRatio =
                    (float) spriteTexture.Width / spriteTexture.Height;
            spriteRectangle.Height =
                    (int) ((spriteRectangle.Width / aspectRatio) + 0.5f);
            xSpeed = displayWidth / ticksToCrossScreen;
            ySpeed = xSpeed;
        }

        public override void StartGame() {
            x = minDisplayX + (initialX * displayWidth);
            y = minDisplayY + (initialY * displayHeight);
            spriteRectangle.X = (int) x;
            spriteRectangle.Y = (int) y;
            base.StartGame();
        }
    }

    public class BatSprite : MovingSprite {
        private PlayerIndex player;

        /// <summary>
        /// Aktualizuje pooenie paki zalenie od stanu pada.
        /// </summary>
        public override void Update(ISpriteBasedGame game) {
            x = x + (xSpeed * game.GetGamePad(player).ThumbSticks.Left.X);
            y = y - (ySpeed * game.GetGamePad(player).ThumbSticks.Left.Y);
            spriteRectangle.X = (int) x;
            spriteRectangle.Y = (int) y;
        }

        public BatSprite(Texture2D inSpriteTexture,
                float widthFactor, float ticksToCrossScreen,
                float inMinDisplayX, float inMaxDisplayX,
                float inMinDisplayY, float inMaxDisplayY,
                float inInitialX, float inInitialY,
                PlayerIndex inPlayer)
        : base(inSpriteTexture, widthFactor, ticksToCrossScreen,
        inMinDisplayX, inMaxDisplayX,
        inMinDisplayY, inMaxDisplayY,
        inInitialX, inInitialY) {
            player = inPlayer;
        }

    }

    public class BallSprite : MovingSprite {
        private SoundEffect BatHitSound;

        private SoundEffect RepeatHitSound;

        private SoundEffect EdgeHitSound;

        private SoundEffect LoseLifeSound;

        private BatSprite player1;
        private BatSprite player2;

        private int player1Score;
        private int player2Score;

        /// <summary>
        /// Aktualizuje pooenie piki. Obsuguje kolizje z pak
        /// </summary>
        /// <param name="game">Gra, w ktrej wystpuje sprajt sera</param>
        public override void Update(ISpriteBasedGame game) {
            x = x + xSpeed;
            y = y + ySpeed;

            spriteRectangle.X = (int) (x + 0.5f);
            spriteRectangle.Y = (int) (y + 0.5f);

            if (player1.CheckCollision(spriteRectangle)) {
                // Pierwszy gracz trafi w pik.
                BatHitSound.Play();
                xSpeed = xSpeed * -1;
            }

            if (player2.CheckCollision(spriteRectangle)) {
                // Drugi gracz trafi w pik.
                BatHitSound.Play();
                xSpeed = xSpeed * -1;
            }

            if (x + spriteRectangle.Width >= maxDisplayX) {
                // Pika zostaa trafiona z prawej strony.
                // Pierwszy gracz zdoby punkty.
                LoseLifeSound.Play();
                player1Score++;
                xSpeed = Math.Abs(xSpeed) * -1;
            }

            if (x <= minDisplayX) {
                // Pika zostaa trafiona z lewej strony.
                // Drugi gracz zdoby punkty.
                LoseLifeSound.Play();
                player2Score++;
                xSpeed = Math.Abs(xSpeed);
            }

            if (y + spriteRectangle.Height >= maxDisplayY) {
                // Pika zderzya si z doln krawdzi.
                EdgeHitSound.Play();
                ySpeed = Math.Abs(ySpeed) * -1;
            }

            if (y <= minDisplayY) {
                // Pika uderzya w grn cz ekranu.
                EdgeHitSound.Play();
                ySpeed = Math.Abs(ySpeed);
            }
            game.SetMessage(player1Score.ToString() + " : " + player2Score.ToString());
        }

        public override void StartGame() {
            player1Score = 0;
            player2Score = 0;
            base.StartGame();
        }

        public BallSprite(Texture2D inSpriteTexture,
                float widthFactor, float ticksToCrossScreen,
                float inMinDisplayX, float inMaxDisplayX,
                float inMinDisplayY, float inMaxDisplayY,
                float inInitialX, float inInitialY,
                BatSprite inPlayer1,
                BatSprite inPlayer2,
                SoundEffect inBatHitSound,
                SoundEffect inRepeatHitSound,
                SoundEffect inEdgeHitSound,
                SoundEffect inLoseLifeSound)
        : base(inSpriteTexture, widthFactor, ticksToCrossScreen,
        inMinDisplayX, inMaxDisplayX,
        inMinDisplayY, inMaxDisplayY,
        inInitialX, inInitialY) {
            player1 = inPlayer1;
            player2 = inPlayer2;
            BatHitSound = inBatHitSound;
            RepeatHitSound = inRepeatHitSound;
            EdgeHitSound = inEdgeHitSound;
            LoseLifeSound = inLoseLifeSound;
        }
    }

#endregion


    /// <summary>
    /// To jest gwny typ Twojej gry.
    /// </summary>
    public class PongGame : Microsoft.Xna.Framework.Game, ISpriteBasedGame {
        GraphicsDeviceManager graphics;
        SpriteBatch spriteBatch;

        // wiat gry

        public BatSprite Player1;
        public BatSprite Player2;
        public BallSprite Cheese;
        public TitleSprite Title;
        public BaseSprite Background;

        public List<BaseSprite> GameSprites = new List<BaseSprite>();

#region Wartoci reprezentujce wymiary ekranu

        int displayWidth;
        int displayHeight;
        float overScanPercentage = 10.0f;
        float minDisplayX;
        float maxDisplayX;
        float minDisplayY;
        float maxDisplayY;

        private void setScreenSizes() {
            displayWidth = graphics.GraphicsDevice.Viewport.Width;
            displayHeight = graphics.GraphicsDevice.Viewport.Height;
            float xOverscanMargin = getPercentage(overScanPercentage, displayWidth) / 2.0f;
            float yOverscanMargin = getPercentage(overScanPercentage, displayHeight) / 2.0f;

            minDisplayX = xOverscanMargin;
            minDisplayY = yOverscanMargin;

            maxDisplayX = displayWidth - xOverscanMargin;
            maxDisplayY = displayHeight - yOverscanMargin;
        }


#endregion

#region Metody pomocnicze

        /// <summary>
        /// Oblicza wartoci procentowe
        /// </summary>
        /// <param name="percentage">procent, ktry ma zosta wyznaczony</param>
        /// <param name="inputValue">warto do przeliczenia</param>
        /// <returns>warto wynikowa</returns>

        float getPercentage(float percentage, float inputValue) {
            return (inputValue * percentage) / 100;
        }

#endregion

#region Rysowanie tekstu

        SpriteFont font;

        string displayMessage = "";

        private void loadFont() {
            font = Content.Load<SpriteFont > ("SpriteFont1");
        }

        /// <summary>
        /// Rysuje tekst na ekranie
        /// </summary>
        /// <param name="text">tekst do wywietlenia</param>
        /// <param name="textColor">kolor tekstu</param>
        /// <param name="x">lewa krawd tekstu</param>
        /// <param name="y">grna krawd tekstu</param>

        void drawText(string text, Color textColor, float x, float y) {
            int layer;
            Vector2 textVector = new Vector2(x - 16, y - 16);

            // Rysuje cie
            Color backColor = new Color(0, 0, 0, 10);
            for (layer = 0; layer < 10; layer++) {
                spriteBatch.DrawString(font, text, textVector, backColor);
                textVector.X--;
                textVector.Y++;
            }

            // Rysuje podstaw znakw
            backColor = new Color(190, 190, 190);
            for (layer = 0; layer < 5; layer++) {
                spriteBatch.DrawString(font, text, textVector, backColor);
                textVector.X--;
                textVector.Y++;
            }

            // Rysuje przedni cz znakw
            spriteBatch.DrawString(font, text, textVector, textColor);
        }

        public void CenterBottomMessage(string displayMessage, Color color) {
            Vector2 messageSize = font.MeasureString(displayMessage);
            float y = displayHeight - messageSize.Y;
            float x = (displayWidth - messageSize.X) / 2;
            drawText(displayMessage, color, x, y);

        }

        public void SetMessage(string message) {
            displayMessage = message;
        }

#endregion


#region Zarzdzanie gr i punktacj

        public enum GameState {
            titleScreen,
            playingGame
        }

        GameState state = GameState.titleScreen;

        public void StartGame() {

            foreach(BaseSprite sprite in GameSprites) {
                sprite.StartGame();
            }

            state = GameState.playingGame;
        }

        public GamePadState GetGamePad(PlayerIndex player) {
            return GamePad.GetState(player);
        }

        public void EndGame() {

            foreach(BaseSprite sprite in GameSprites) {
                sprite.EndGame();
            }

            state = GameState.titleScreen;
        }

        public GameState GetState() {
            return state;
        }

#endregion

        public PongGame() {
            graphics = new GraphicsDeviceManager(this);
            Content.RootDirectory = "Content";
        }

        /// <summary>
        /// Umoliwia ewentualn inicjalizacj przed uruchomieniem waciwej gry.
        /// W tym miejscu mona odnale wszystkie wymagane zasoby i zaadowa treci
        /// related content. Wywoanie metody base.Initialize spowoduje przeszukanie wszystkich komponentw
        /// i ich inicjalizacj.
        /// </summary>
        protected override void Initialize() {
            setScreenSizes();
            base.Initialize();
        }

        /// <summary>
        /// Metoda LoadContent jest wywoywana tylko raz dla caej gry i jako taka jest waciwym miejscem
        /// dla kodu adujcego tre.
        /// </summary>
        protected override void LoadContent() {
            // Tworzy nowy obiekt klasy SpriteBatch, ktrego mona uywa do rysowania tekstur.
            spriteBatch = new SpriteBatch(GraphicsDevice);

            Background = new BaseSprite(
                    Content.Load<Texture2D > ("Images/Background"),
                    new Rectangle(0, 0, displayWidth, displayHeight));

            GameSprites.Add(Background);

            Title = new TitleSprite(
                    Content.Load<Texture2D > ("Images/Title"),
                    new Rectangle(0, 0, displayWidth, displayHeight));

            Player1 = new BatSprite(
                    Content.Load<Texture2D > ("Images/WhiteBread"),
                    0.166f, // szeroko chleba wynosi 0,166 szerokoci ekranu
                    150, // pokonanie ekranu zajmuje 150 taktw
                    minDisplayX, maxDisplayX, minDisplayY, maxDisplayY,
                    0.25f, // jedna czwarta szerokoci ekranu od lewej krawdzi
                    0.5f, // rodek ekranu (w pionie)
                    PlayerIndex.One); // sterowany przez pierwszego gracza

            GameSprites.Add(Player1);

            Player2 = new BatSprite(
                    Content.Load<Texture2D > ("Images/BrownBread"),
                    0.166f, // szeroko chleba wynosi 0,166 szerokoci ekranu
                    150, // pokonanie ekranu zajmuje 150 taktw
                    minDisplayX, maxDisplayX, minDisplayY, maxDisplayY,
                    0.75f, // jedna czwarta szerokoci ekranu od prawej krawdzi
                    0.5f, // rodek ekranu (w pionie)
                    PlayerIndex.Two); // sterowany przez drugiego gracza

            GameSprites.Add(Player2);

            Cheese = new BallSprite(
                    Content.Load<Texture2D > ("Images/Cheese"),
                    0.07f, // szeroko sera wynosi 0,07 szerokoci ekranu
                    200, // ser potrzebuje 200 taktw na pokonanie caego ekranu
                    minDisplayX, maxDisplayX, minDisplayY, maxDisplayY,
                    0.25f, // jedna czwarta szerokoci ekranu od lewej krawdzi
                    0.25f, // jedna czwarta wysokoci ekranu od grnej krawdzi
                    Player1, // pierwszy gracz
                    Player2, // drugi gracz
                    Content.Load<SoundEffect > ("Sounds/BreadHit"),
                    Content.Load<SoundEffect > ("Sounds/RepeatHit"),
                    Content.Load<SoundEffect > ("Sounds/EdgeHit"),
                    Content.Load<SoundEffect > ("Sounds/LoseLife"));

            GameSprites.Add(Cheese);

            loadFont();
        }

        /// <summary>
        /// Metoda UnloadContent jest wywoywana tylko raz dla caej gry i jako taka jest waciwym miejscem
        /// dla kodu usuwajcego ca tre z pamici.
        /// </summary>
        protected override void UnloadContent() {
            // TODO: Naley usun ca tre, ktr nie zarzdza ContentManager.
        }

        /// <summary>
        /// Umoliwia grze wykonywanie logiki zwizanej z aktualizacj wiata gry,
        /// sprawdzaniem kolizji, pobieraniem danych wejciowych czy odtwarzaniem dwikw.
        /// </summary>
        /// <param name="gameTime">Udostpnia wartoci reprezentujce biecy czas.</param>
        protected override void Update(GameTime gameTime) {
            // Pozwala opuci gr
            if (GamePad.GetState(PlayerIndex.One).Buttons.Back == ButtonState.Pressed)
                this.Exit();

            switch (state) {
                case GameState.titleScreen:
                    Title.Update(this);
                    break;
                case GameState.playingGame:
                    foreach(BaseSprite sprite in GameSprites)
                {
                    sprite.Update(this);
                }
                    break;
            }

            base.Update(gameTime);
        }

        /// <summary>
        /// Ta metoda jest wywoywana w momencie, w ktrym gra musi narysowa swj wiat.
        /// </summary>
        /// <param name="gameTime">Udostpnia wartoci reprezentujce biecy czas.</param>
        protected override void Draw(GameTime gameTime) {
            spriteBatch.Begin();

            switch (state) {
                case GameState.titleScreen:
                    Title.Draw(spriteBatch);
                    break;
                case GameState.playingGame:
                    foreach(BaseSprite sprite in GameSprites)
                {
                    sprite.Draw(spriteBatch);
                }
                    break;
            }

            CenterBottomMessage(displayMessage, Color.Red);

            spriteBatch.End();

            base.Draw(gameTime);
        }
    }
}
